/**************************************************************************
 *
 * Copyright 2010, 2011 BMW Car IT GmbH
 * Copyright (C) 2011 DENSO CORPORATION and Robert Bosch Car Multimedia Gmbh
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <iostream>
#include <sstream>
#include <signal.h>
#include <unistd.h>
#include "WLContext.h"
#include "WLEGLSurface.h"
#include "gles2application.h"
#include "ilm_control.h"

#define LAYER_ID 1000
#define SURFACE_ID 5100
#define WIDTH 1024
#define HEIGHT 768
#define DEF_X 0
#define DEF_Y 0
#define ANIM_TIME 33
#define MAX_LAYER_ID 10000
#define MAX_SURFACE_ID 10000
#define MAX_WIDTH 1024
#define MAX_HEIGHT 768
#define MAX_ANIM_TIME 1000
#define MAX_X 1024
#define MAX_Y 768

using namespace std;

int gRunLoop = 0;
bool touched = false;

/*test notification defines*/
int user_data = 3;

void notificationCB(ilmObjectType object,
                    t_ilm_uint id,
                    t_ilm_bool created,
                    void* user_data)
{
    int* ud = (int*) user_data;
    printf("ILM object: %s(%u) %s, with user_data: %d\n",
            (object) ? "Layer":"Surface",
            id,
            (created) ? "created":"deleted",
            *ud);

    *ud = *ud-1;
}

ilmErrorTypes test_init_object_notification();
ilmErrorTypes test_init_object_notification()
{
    ilmErrorTypes ilmret = ILM_SUCCESS;

    ilmret = ilm_registerNotification(&notificationCB, (void*) &user_data);
    if (ilmret != ILM_SUCCESS)
    {
        printf("Error by ilm_registerNotification\n");
        goto error;
    }

    ilmret = ilm_registerNotification(&notificationCB, (void*) &user_data);
    if (ilmret == ILM_SUCCESS)
    {
        printf("Error by ilm_registerNotification: %u, second initialization should not work\n",ilmret);
        goto error;
    }
    return ILM_SUCCESS;

error:
    return ILM_FAILED;
}

ilmErrorTypes test_recheck_object_notification();
ilmErrorTypes test_recheck_object_notification()
{
    ilmErrorTypes ilmret = ILM_SUCCESS;
    if (user_data == 0)
    {
        ilmret = ilm_unregisterNotification();
        if (ilmret != ILM_SUCCESS)
        {
            printf("ERROR by ilm_unregisterNotification\n");
            goto error;
        }

        ilmret = ilm_unregisterNotification();
        if (ilmret == ILM_SUCCESS)
        {
            printf("ERROR by ilm_unregisterNotification, second unregistration should not work\n");
            goto error;
        }

        /*register notification again*/
        user_data = 5;
        ilmErrorTypes ilmret = ilm_registerNotification(&notificationCB, (void*) &user_data);
        if (ilmret != ILM_SUCCESS)
        {
            printf("Error by ilm_registerNotification\n");
            goto error;
        }

        ilmret = ilm_registerNotification(&notificationCB, (void*) &user_data);
        if (ilmret == ILM_SUCCESS)
        {
            printf("Error by ilm_registerNotification: %u, second initialization should not work\n",ilmret);
            goto error;
        }
    }

    return ILM_SUCCESS;
error:
    return ILM_FAILED;
}
static void SigFunc(int)
{
    printf("Caught signal\n");
    gRunLoop = 0;
}

void usage(string& name)
{
    cerr << "Expected Arguments for " << name << endl;
    cerr << "Usage: " 
         << name << " [options]" << endl << "     "
         "Where options are:" << endl << "     "
         "-l<layer ID> (default " << LAYER_ID << ", "
         "max " << MAX_LAYER_ID << ")" << endl << "     "
         "-s<surface ID> (default " << SURFACE_ID << ", "
         "max " << MAX_SURFACE_ID << ")" << endl << "     "
         "-w<surface width> (default " << WIDTH << ", "
         "max " << MAX_WIDTH << ")" << endl << "     "
         "-h<surface height> (default " << HEIGHT << ", "
         "max " << MAX_HEIGHT << ")" << endl << "     "
         "-x<x surface position> (default " << DEF_X << ", "
         "max " << MAX_WIDTH << ")" << endl << "     "
         "-y<y surface position> (default " << DEF_Y << ", "
         "max " << MAX_HEIGHT << ")" << endl << "     "
         "-W<layer width> (default " << WIDTH << ", "
         "max " << MAX_WIDTH << ")" << endl << "     "
         "-H<layer height> (default " << HEIGHT << ", "
         "max " << MAX_HEIGHT << ")" << endl << "     "
         "-X<x layer position> (default " << DEF_X << ", "
         "max " << MAX_WIDTH << ")" << endl << "     "
         "-Y<y layer position> (default " << DEF_Y << ", "
         "max " << MAX_HEIGHT << ")" << endl << "     "
         "-t<animation time> (default " << ANIM_TIME << ", "
         "max " << MAX_ANIM_TIME << ")" << endl;
}

struct testConfiguration
{
    unsigned int layerID;
    unsigned int surfaceID;
    unsigned int surfaceWidth;
    unsigned int surfaceHeight;
    unsigned int surfaceX;
    unsigned int surfaceY;
    unsigned int layerWidth;
    unsigned int layerHeight;
    unsigned int layerX;
    unsigned int layerY;
    unsigned int animationTime;
    unsigned int x;
    unsigned int y;
};

void setDefaultConfiguration(testConfiguration& config)
{
    config.layerID = LAYER_ID;
    config.surfaceID = SURFACE_ID;
    config.surfaceWidth = WIDTH;
    config.surfaceHeight = HEIGHT;
    config.surfaceX = DEF_X;
    config.surfaceY = DEF_Y;
    config.layerWidth = WIDTH;
    config.layerHeight = HEIGHT;
    config.layerX = DEF_X;
    config.layerY = DEF_Y;
    config.animationTime = ANIM_TIME;
}

bool parseCommandLine(int argc, const char* argv[], testConfiguration& config)
{
    string exeName(argv[0]);
    bool result = true;

    while ((argc > 1) && (argv[1][0] == '-'))
    {
        switch (argv[1][1])
        {
            case 'l':
                {
                    string layerID(&argv[1][2]);
                    istringstream iss(layerID);
                    unsigned short int ID;
                    iss >> ID;
                    if (!iss.fail() && (ID <= MAX_LAYER_ID))
                    {
                        config.layerID = ID;
                    }
                    else
                    {
                        result = false;
                    }
                }
            break;

            case 's':
                {
                    string surfaceID(&argv[1][2]);
                    istringstream iss(surfaceID);
                    unsigned short int ID;
                    iss >> ID;
                    if (!iss.fail() && (ID <= MAX_SURFACE_ID))
                    {
                        config.surfaceID = ID;
                    }
                    else
                    {
                        result = false;
                    }
                }
            break;

            case 'w':
                {
                    string width(&argv[1][2]);
                    istringstream iss(width);
                    unsigned short int widthVal;
                    iss >> widthVal;
                    if (!iss.fail() && (widthVal <= MAX_WIDTH))
                    {
                        config.surfaceWidth = widthVal;
                    }
                    else
                    {
                        result = false;
                    }
                }
            break;

            case 'h':
                {
                    string height(&argv[1][2]);
                    istringstream iss(height);
                    unsigned short int heightVal;
                    iss >> heightVal;
                    if (!iss.fail() && (heightVal <= MAX_HEIGHT))
                    {
                        config.surfaceHeight = heightVal;
                    }
                    else
                    {
                        result = false;
                    }
                }
            break;

            case 't':
                {
                    string time(&argv[1][2]);
                    istringstream iss(time);
                    unsigned short int timeVal;
                    iss >> timeVal;
                    if (!iss.fail() && (timeVal <= MAX_ANIM_TIME))
                    {
                        config.animationTime = timeVal;
                    }
                    else
                    {
                        result = false;
                    }
                }
            break;

            case 'x':
                {
                    string x(&argv[1][2]);
                    istringstream iss(x);
                    unsigned short int xVal;
                    iss >> xVal;
                    if (!iss.fail() && (xVal <= MAX_X))
                    {
                        config.surfaceX = xVal;
                    }
                    else
                    {
                        result = false;
                    }
                }
            break;

            case 'y':
                {
                    string y(&argv[1][2]);
                    istringstream iss(y);
                    unsigned short int yVal;
                    iss >> yVal;
                    if (!iss.fail() && (yVal <= MAX_Y))
                    {
                        config.surfaceY = yVal;
                    }
                    else
                    {
                        result = false;
                    }
                }
            break;

            case 'W':
                {
                    string width(&argv[1][2]);
                    istringstream iss(width);
                    unsigned short int widthVal;
                    iss >> widthVal;
                    if (!iss.fail() && (widthVal <= MAX_WIDTH))
                    {
                        config.layerWidth = widthVal;
                    }
                    else
                    {
                        result = false;
                    }
                }
            break;

            case 'H':
                {
                    string height(&argv[1][2]);
                    istringstream iss(height);
                    unsigned short int heightVal;
                    iss >> heightVal;
                    if (!iss.fail() && (heightVal <= MAX_HEIGHT))
                    {
                        config.layerHeight = heightVal;
                    }
                    else
                    {
                        result = false;
                    }
                }
            break;

            case 'X':
                {
                    string x(&argv[1][2]);
                    istringstream iss(x);
                    unsigned short int xVal;
                    iss >> xVal;
                    if (!iss.fail() && (xVal <= MAX_X))
                    {
                        config.layerX = xVal;
                    }
                    else
                    {
                        result = false;
                    }
                }
            break;

            case 'Y':
                {
                    string y(&argv[1][2]);
                    istringstream iss(y);
                    unsigned short int yVal;
                    iss >> yVal;
                    if (!iss.fail() && (yVal <= MAX_Y))
                    {
                        config.layerY = yVal;
                    }
                    else
                    {
                        result = false;
                    }
                }
            break;

            default:
                result = false;
        }

        ++argv;
        --argc;
    }

    return result;

}


int main(int argc, char **argv)
{
    WLContext* wlContext;
    WLEGLSurface* eglSurface;

    testConfiguration config;
    setDefaultConfiguration(config);

    if (!parseCommandLine(argc,(const char **)argv, config))
    {
        string exeName(argv[0]);
        usage(exeName);
        exit(1);
    }

    // signal handling
    signal(SIGINT,SigFunc);
    signal(SIGTERM,SigFunc);

    ilm_init();

    wlContext = new WLContext();
    wlContext->InitWLContext(NULL, NULL, &TouchListener);

    eglSurface = new WLEGLSurface(wlContext);
    eglSurface->CreateSurface(config.surfaceWidth, config.surfaceHeight);
    eglSurface->CreateIlmSurface(&config.layerID,
                                 &config.surfaceID,
                                 config.surfaceWidth,
                                 config.surfaceHeight,
                                 config.surfaceX, config.surfaceY,
                                 config.layerWidth,
                                 config.layerHeight,
                                 config.layerX, config.layerY);

    if (!InitRenderer(config.surfaceWidth, config.surfaceHeight)){
        fprintf(stderr, "Failed to init renderer\n");
        return -1;
    }

    gRunLoop = 1;

    test_init_object_notification();

    while (gRunLoop)
    {
        if (!touched)
        {
            CheckForEvents(wlContext->GetWLDisplay());
        }
        else
        {
            WaitForEvents(wlContext->GetWLDisplay());
        }
        test_recheck_object_notification();
        draw(config.animationTime, eglSurface);
        usleep(1000);
    }

    ilm_layerRemoveSurface(config.layerID, config.surfaceID);

    ilm_commitChanges();
    eglSurface->DestroyIlmSurface();
    sleep(3);
    destroyGlApplication();
    ilm_destroy();
    delete eglSurface;
    delete wlContext;
    return 0;
}
